<?php

/**
 * FXH universal password checker — mirrors modern WordPress logic.
 */
function fxh_check_password(string $password, string $hash): bool
{
    $password = (string)$password;
    $hash     = trim((string)$hash);

    if ($password === '' || $hash === '') {
        return false;
    }

    // Legacy very-old MD5 hashes (WP still supports them internally)
    if (strlen($hash) <= 32) {
        return hash_equals($hash, md5($password));
    }

    // Safety — same limit WP uses
    if (strlen($password) > 4096) {
        return false;
    }

    // Modern WordPress / WooCommerce hashes with `$wp` prefix.
    // WordPress does:
    //   $password_to_verify = base64_encode( hash_hmac( 'sha384', $password, 'wp-sha384', true ) );
    //   password_verify( $password_to_verify, substr( $hash, 3 ) );
    if (strpos($hash, '$wp') === 0) {
        $derived = base64_encode(hash_hmac('sha384', $password, 'wp-sha384', true));
        $bcrypt  = substr($hash, 3); // strip `$wp`
        return password_verify($derived, $bcrypt);
    }

    // Older portable hashes starting with $P$ (phpass).
    if (strpos($hash, '$P$') === 0) {
        // Use the small compatible hasher we added earlier.
        // If you don't have wp-password.php, reuse the class we used before.
        require_once __DIR__ . '/wp-password.php';
        if (class_exists('WP_PasswordHash')) {
            $hasher = new WP_PasswordHash(8, true);
            return $hasher->CheckPassword($password, $hash);
        }
    }

    // Fallback for plain bcrypt/argon2 (no prefix)
    return password_verify($password, $hash);
}
